<?php

namespace Database\Seeders;

use App\Models\District;
use App\Models\SubDistrict;
use App\Models\Village;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\File;

class LocationDataSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info("Starting LocationDataSeeder...");

        // 1. Create District (Kabupaten)
        try {
            $district = District::firstOrCreate(['name' => 'Mempawah']);
            $this->command->info("District: " . $district->name);
        } catch (\Exception $e) {
            $this->command->error("Error creating district: " . $e->getMessage());
            return;
        }

        // 2. Read JSON Data
        $path = public_path('batas-desa.json');
        if (!File::exists($path)) {
            $this->command->error("File not found: $path");
            return;
        }

        $data = json_decode(File::get($path), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->command->error("JSON Error: " . json_last_error_msg());
            return;
        }
        $features = $data['features'] ?? [];

        $this->command->info("Seeding " . count($features) . " villages...");

        foreach ($features as $feature) {
            try {
                $props = $feature['properties'];
                $geom = $feature['geometry'];

                $kecamatanName = $props['Kecamatan'] ?? 'Unknown';
                $villageName = $props['Nama'] ?? 'Unknown';

                // 3. Create/Get SubDistrict (Kecamatan)
                $subDistrictId = \DB::table('sub_districts')->where('name', $kecamatanName)->value('id');
                if (!$subDistrictId) {
                    $subDistrictId = \DB::table('sub_districts')->insertGetId([
                        'name' => $kecamatanName,
                        'district_id' => $district->id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }

                // 4. Create Village (Desa)
                $villageExists = \DB::table('villages')->where('name', $villageName)->where('sub_district_id', $subDistrictId)->exists();
                if (!$villageExists) {
                    \DB::table('villages')->insert([
                        'name' => $villageName,
                        'sub_district_id' => $subDistrictId,
                        'coordinates' => json_encode($geom),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }

                $this->command->info("Seeded: $villageName ($kecamatanName)");
            } catch (\Exception $e) {
                $this->command->error("Error seeding village: " . ($props['Nama'] ?? 'Unknown') . " - " . $e->getMessage());
                // Rethrow to see full error in console
                throw $e;
            }
        }

        $this->command->info("Seeding completed.");
    }
}
